--[[ 
****************************************************
* Historian add-on for World of Warcraft.
* 
* Collects various interesting but ultimately useless statistics about your WoW characters.
****************************************************
]]

local L = AceLibrary("AceLocale-2.2"):new("Historian");
local LS = AceLibrary("AceLocale-2.2"):new("Historian-Shared");
local abacus = AceLibrary("Abacus-2.0");
local f = AceLibrary("HistorianFormatting-1.0");

local DEBUG = false;


-- *************************************
-- * Chat commands
-- *************************************
local HistorianOptions = { 
    type='group',
    args = {
        show = {
			type = 'execute',
			name = LS["CMD_GUI_TITLE"],
			desc = LS["CMD_GUI_DESC"],
			func = "ToggleWindow",
        },
        firstUse = {
			type = 'execute',
			name = LS["CMD_FIRST_USE_TITLE"],
			desc = LS["CMD_FIRST_USE_DESC"],
			func = "FirstUse",
        },
		showTooltip = {
			type = "toggle",
			name = LS["CMD_SHOWTOOLTIP_TITLE"],
			desc = LS["CMD_SHOWTOOLTIP_DESC"],
			get = "IsShowingTooltip",
			set = "ToggleShowingTooltip",
		},
		countPartyKills = {
			type = "toggle",
			name = LS["CMD_COUNTPARTYKILLS_TITLE"],
			desc = LS["CMD_COUNTPARTYKILLS_DESC"],
			get = "IsCountingPartyKills",
			set = "ToggleCountingPartyKills",
		},
		countInstanceKills = {
			type = "toggle",
			name = LS["CMD_COUNTINSTANCEKILLS_TITLE"],
			desc = LS["CMD_COUNTINSTANCEKILLS_DESC"],
			get = "IsCountingInstanceKills",
			set = "ToggleCountingInstanceKills",
		},
    },
}


-- *************************************
-- * Set up Ace
-- *************************************
Historian = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceEvent-2.0", "AceDB-2.0", "AceHook-2.1");

Historian:RegisterChatCommand(L["SLASH-COMMANDS"], HistorianOptions);

Historian:RegisterDB("HistorianDB", "HistorianCharDB");


-- *************************************
-- * Set up defaults
-- *************************************
Historian:RegisterDefaults("profile", {
	showTooltip = true,
	countPartyKills = true,
	countInstanceKills = true,
} )

Historian:RegisterDefaults("char", {
	Loot = { 
		MostMoneyEver = 0,
		ItemsOfQuality = { 0, 0, 0, 0, 0, 0 },
	},
	Advancement = {
		Levels = { },
		RecipesLearned = 0,
		TotalDeaths = 0,
		QuestsCompleted = 0,
		Tradeskills = { },
	},
	Discoveries = {
		Areas = 0,
	},
	EnvironmentalDamage = {
		Drowning = {
			Times = 0,
			TotalDamage = 0,
		},
		Fatique = {
			Times = 0,
			TotalDamage = 0,
		},
		Falling = {
			Times = 0,
			TotalDamage = 0,
		},
		Fire = {
			Times = 0,
			TotalDamage = 0,
		},
		Lava = {
			Times = 0,
			TotalDamage = 0,
		},
		Slime = {
			Times = 0,
			TotalDamage = 0,
		},
	},
	Follow = { 
		Players = {
		},
		MostFollowed = {
			Count = 0,
			Name = "",
		}
	},
	Kills = {
		Favorite = {
			Count = 0,
			Name = "",
		},
		Mobs = {
		},
		Total = 0,
		TotalKillingBlows = 0,
	},
	Consumption = {
		Food = 0,
		Drink = 0,
		Tipsy = 0,
		Drunk = 0,
		Smashed = 0,
	},
	Spells = {
		Favorite = {
			Count = 0,
			Name = "",
		},
		Casts = {
		},
		Total = 0,
	},
	PvP = {
	},
	Jumps = 0,
	AFK = 0,
	FirstUse = nil,
	DataVersion = 0,
} )


-- *************************************
-- * Properties
-- *************************************
function Historian:IsShowingTooltip()
	return self.db.profile.showTooltip;
end
 
function Historian:ToggleShowingTooltip()
	self.db.profile.showTooltip = not self.db.profile.showTooltip;
end


function Historian:IsCountingPartyKills()
	return self.db.profile.countPartyKills;
end
 
function Historian:ToggleCountingPartyKills()
	self.db.profile.countPartyKills = not self.db.profile.countPartyKills;
end


function Historian:IsCountingInstanceKills()
	return self.db.profile.countInstanceKills;
end
 
function Historian:ToggleCountingInstanceKills()
	self.db.profile.countInstanceKills = not self.db.profile.countInstanceKills;
end


-- *************************************
-- * DebugPrint
-- * Prints a chat message, provided the DEBUG variable is set to true
-- *************************************
function Historian:DebugPrint(message)
	if (DEBUG) then
		self:Print("(DEBUG) " .. message);
	end;
end;


-- *************************************
-- * OnInitialize
-- * Fires when the addon is first loaded
-- *************************************
function Historian:OnInitialize()
	self:MigrateData();	

	self.isEating = false;
	self.isDrinking = false;
	
	if (self.db.char.FirstUse == nil) then
		self.db.char.FirstUse = date();
	end;
end;


-- *************************************
-- * OnEnable
-- * Fires when the addon is enabled
-- *************************************
function Historian:OnEnable()
	self:RegisterForEvents();
	self:CreateSpellTable();
end;


-- *************************************
-- * CreateSpellTable
-- * Prepares the self.realSpells variable for use in UNIT_SPELLCAST_SUCCEEDED
-- *************************************
function Historian:CreateSpellTable()
	self.realSpells = {};

	local i, s;
	-- Starts at 2 in order to exclude the General tab
	for i = 2, MAX_SKILLLINE_TABS do
		local name, texture, offset, numSpells = GetSpellTabInfo(i);
		if not name then break; end;

		for s = offset + 1, offset + numSpells do
			local spell, rank = GetSpellName(s, BOOKTYPE_SPELL);
			self.realSpells[spell] = true;
		end;
	end;
end;


-- *************************************
-- * ToggleWindow
-- *************************************
function Historian:ToggleWindow()
	if (HistorianTabContainer == nil) then
		return self:DebugPrint("The Historian window did not load correctly and cannot be displayed.");
	end;
	
	if (HistorianTabContainer.IsShowing) then
		HistorianTabContainer:Hide();
	else
		HistorianTabContainer:Show();
	end;
end;



-- *************************************
-- * FirstUse
-- *************************************
function Historian:FirstUse()
	self:Print(L["MSG_FIRST_USE"], UnitName("player"), self.db.char.FirstUse);
end;


-- *************************************
-- * OnMoneyChanged
-- *************************************
function Historian:OnMoneyChanged()
	local highestAmount = self.db.char.Loot.MostMoneyEver;
	local currentMoney = GetMoney();

	if (currentMoney > highestAmount) then
		self.db.char.Loot.MostMoneyEver = currentMoney;
		self:TriggerEvent("Historian_Updated");
	end;
end;


-- *************************************
-- * RecordItemLooted (Player received an item)
-- *************************************
function Historian:RecordItemLooted(item)
	if (item == nil) then return; end;
	
	local itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount, itemEquipLoc, itemTexture = GetItemInfo(item);

	if (itemRarity > 0) then
		local totalLooted = f:AsNumber(self.db.char.Loot.ItemsOfQuality[itemRarity]) + 1;
		self.db.char.Loot.ItemsOfQuality[itemRarity] = totalLooted;
		self:TriggerEvent("Historian_Updated");
	end;
end;


-- *************************************
-- * GetKillStatsForEnemy
-- *************************************
function Historian:GetKillStatsForEnemy(mobName)
	return self.db.char.Kills.Mobs[mobName];
end;


-- *************************************
-- * RecordKill (Player killed something)
-- *************************************
function Historian:RecordKill(mobKilled, killingBlow)
	if (mobKilled ~= nil) then
		if (self.db.char.Kills.Mobs[mobKilled] == nil) then
			self.db.char.Kills.Mobs[mobKilled] = {};
			self.db.char.Kills.Mobs[mobKilled].KillingBlows = 0;
			self.db.char.Kills.Mobs[mobKilled].Any = 0;
		end;
		
		local updatedCount = f:AsNumber(self.db.char.Kills.Mobs[mobKilled].Any) + 1;
		self.db.char.Kills.Mobs[mobKilled].Any = updatedCount;
		
		local favoritePreyKills = self.db.char.Kills.Favorite.Count;
		
		if (updatedCount > favoritePreyKills) then
			self.db.char.Kills.Favorite.Count = updatedCount;

			local favoritePreyName = self.db.char.Kills.Favorite.Name;

			if (favoritePreyName ~= mobKilled) then
				self.db.char.Kills.Favorite.Name = mobKilled;
			end;
		end;
		
		local totalKills = f:AsNumber(self.db.char.Kills.Total) + 1;
		self.db.char.Kills.Total = totalKills;

		if (killingBlow) then
			local updatedKbCount = f:AsNumber(self.db.char.Kills.Mobs[mobKilled].KillingBlows) + 1;
			self.db.char.Kills.Mobs[mobKilled].KillingBlows = updatedKbCount;
			local totalKBs = f:AsNumber(self.db.char.Kills.TotalKillingBlows) + 1;
			self.db.char.Kills.TotalKillingBlows = totalKBs;
		end;
		
		self:TriggerEvent("Historian_Updated");
	end;
end;


-- *************************************
-- * OnDeath
-- *************************************
function Historian:OnDeath()
	local totalDeaths = f:AsNumber(self.db.char.Advancement.TotalDeaths) + 1;

	self.db.char.Advancement.TotalDeaths = totalDeaths;

	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * RecordLevelUp
-- *************************************
function Historian:RecordLevelUp(newLevel, levelInfo)
	self.db.char.Advancement.Levels[newLevel] = levelInfo;
	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * OnFollow (Player started following another player)
-- *************************************
function Historian:OnFollow(playerFollowed)
	local times = f:AsNumber(self.db.char.Follow.Players[playerFollowed]) + 1;
	local currentMax = f:AsNumber(self.db.char.Follow.MostFollowed.Count);
	
	self.db.char.Follow.Players[playerFollowed] = times;

	if (times > currentMax) then
		self.db.char.Follow.MostFollowed.Name = playerFollowed;
		self.db.char.Follow.MostFollowed.Count = times;
		self:TriggerEvent("Historian_Updated");
	end;
end;

	
-- *************************************
-- * OnQuestCompleted (Player completed yet another quest :)
-- *************************************
function Historian:OnQuestCompleted()
	local questsCompleted = f:AsNumber(self.db.char.Advancement.QuestsCompleted) + 1;
	
	self.db.char.Advancement.QuestsCompleted = questsCompleted;
	
	self:TriggerEvent("Historian_Updated");
end;



-- *************************************
-- * OnRecipeLearned (Player learned a new recipe)
-- *************************************
function Historian:OnRecipeLearned()
	local totalKnown = f:AsNumber(self.db.char.Advancement.RecipesLearned) + 1;
	
	self.db.char.Advancement.RecipesLearned = totalKnown;
	
	self:TriggerEvent("Historian_Updated");
end;
	

-- *************************************
-- * OnDiscovery (Player discovered a new zone/area)
-- *************************************
function Historian:OnDiscovery()
	local totalDiscovered = f:AsNumber(self.db.char.Discoveries.Areas) + 1;
	
	self.db.char.Discoveries.Areas = totalDiscovered;
	
	self:TriggerEvent("Historian_Updated");
end;

	
-- *************************************
-- * Environmental damage
-- *************************************
function Historian:RegisterEnvironmentDamage(type, damage)
	local times = f:AsNumber(self.db.char.EnvironmentalDamage[type].Times) + 1;
	local totalDamage = f:AsNumber(self.db.char.EnvironmentalDamage[type].TotalDamage) + f:AsNumber(damage);
	
	self.db.char.EnvironmentalDamage[type].Times = times;
	self.db.char.EnvironmentalDamage[type].TotalDamage = totalDamage;

	self:TriggerEvent("Historian_Updated");
end;
	
	
-- *************************************
-- * UpdateRecipeCount
-- *************************************
function Historian:UpdateRecipeCount()
	local total = 0;
	
	for key, value in pairs(self.db.char.Advancement.Tradeskills) do
		total = total + f:AsNumber(value["Total"]);
	end;

	self.db.char.Advancement.RecipesLearned = total;
	self:TriggerEvent("Historian_Updated");
end;



-- *************************************
-- * OnEating
-- *************************************
function Historian:OnEating()
	self.db.char.Consumption.Food = self.db.char.Consumption.Food + 1;
	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * OnDrinking
-- *************************************
function Historian:OnDrinking()
	self.db.char.Consumption.Drink = self.db.char.Consumption.Drink + 1;
	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * OnJump
-- *************************************
function Historian:OnJump()
	self.db.char.Jumps = self.db.char.Jumps + 1;
	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * OnPlayerTipsy
-- *************************************
function Historian:OnPlayerTipsy()
	self.db.char.Consumption.Tipsy = self.db.char.Consumption.Tipsy + 1;
	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * OnPlayerDrunk
-- *************************************
function Historian:OnPlayerDrunk()
	self.db.char.Consumption.Drunk = self.db.char.Consumption.Drunk + 1;
	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * OnPlayerCompletelySmashed
-- *************************************
function Historian:OnPlayerCompletelySmashed()
	self.db.char.Consumption.Smashed = self.db.char.Consumption.Smashed + 1;
	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * OnPlayerAFK
-- *************************************
function Historian:OnPlayerAFK()
	self.db.char.AFK = self.db.char.AFK + 1;
	self:TriggerEvent("Historian_Updated");
end;


-- *************************************
-- * RecordSpellcast (Player cast a spell)
-- *************************************
function Historian:RecordSpellcast(spellName, isRealSpell)
	if (spellName ~= nil) then
		if (self.db.char.Spells.Casts[spellName] == nil) then
			self.db.char.Spells.Casts[spellName] = {};
			self.db.char.Spells.Casts[spellName].Succeeded = 0;
		end;
		
		local updatedCount = f:AsNumber(self.db.char.Spells.Casts[spellName].Succeeded) + 1;
		self.db.char.Spells.Casts[spellName].Succeeded = updatedCount;
		self:DebugPrint("You've cast " .. spellName .. " a total of " .. f:IntToStr(updatedCount) .. " time(s).");

		if not isRealSpell then
			-- This is not really a spell, so it cannot be a favorite
			self:DebugPrint("This is not a usable spell, so it will not count towards the total and it cannot be a favorite.");
			return;
		end;
		
		local favoriteCount = self.db.char.Spells.Favorite.Count;
		
		if (updatedCount > favoriteCount) then
			self.db.char.Spells.Favorite.Count = updatedCount;
			self.db.char.Spells.Favorite.Name = spellName;
			self:DebugPrint(spellName .. " is now your favorite spell.");
		end;
		
		local updatedTotal = f:AsNumber(self.db.char.Spells.Total) + 1;
		self.db.char.Spells.Total = updatedTotal;
		self:DebugPrint("You have cast a total of " .. f:IntToStr(updatedTotal) .. " spell(s).");
		self:TriggerEvent("Historian_Updated");
	end;
end;



-- *************************************
-- * GetTradeskillStats
-- *************************************
function Historian:GetTradeskillStats(tradeSkillName)
	if (self.db.char.Advancement.Tradeskills == nil) then
		self.db.char.Advancement.Tradeskills = {};
	end;
	
	if (self.db.char.Advancement.Tradeskills[tradeSkillName] == nil) then
		self.db.char.Advancement.Tradeskills[tradeSkillName] = { Total = 0, WithHeaders = 0 };
	end;		

	local data = self.db.char.Advancement.Tradeskills[tradeSkillName];
	return data.Total, data.WithHeaders;
end;


-- *************************************
-- * SetTradeskillStats
-- *************************************
function Historian:SetTradeskillStats(tradeSkillName, total, withHeaders)
	self.db.char.Advancement.Tradeskills[tradeSkillName] = { Total = total, WithHeaders = withHeaders };
end;


-- *************************************
-- * GetHabitStats
-- *************************************
function Historian:GetHabitStats()
	local result = {};
	
	result.MostFollowed = { Name = self.db.char.Follow.MostFollowed.Name, Count = self.db.char.Follow.MostFollowed.Count };
	result.FavoritePrey = { Name = self.db.char.Kills.Favorite.Name, Count = self.db.char.Kills.Favorite.Count };
	result.FavoriteSpell = { Name = self.db.char.Spells.Favorite.Name, Count = self.db.char.Spells.Favorite.Count };
	result.Food = self.db.char.Consumption.Food;
	result.Drink = self.db.char.Consumption.Drink;
	result.Tipsy = self.db.char.Consumption.Tipsy;
	result.Drunk = self.db.char.Consumption.Drunk;
	result.Smashed = self.db.char.Consumption.Smashed;
	result.AFK = self.db.char.AFK;
	result.Jumps = self.db.char.Jumps;
	
	return result;
end;


-- *************************************
-- * GetAchievementStats
-- *************************************
function Historian:GetAchievementStats()
	local result = {};
	
	result.QuestsCompleted = self.db.char.Advancement.QuestsCompleted;
	result.RecipesLearned = self.db.char.Advancement.RecipesLearned;
	result.AreasDiscovered = self.db.char.Discoveries.Areas;
	result.TotalSpellsCast = self.db.char.Spells.Total;
	result.TotalKills = self.db.char.Kills.Total;
	
	return result;
end;


-- *************************************
-- * GetDisappointmentStats
-- *************************************
function Historian:GetDisappointmentStats()
	local result = {};

	result.Drowning = { Times = self.db.char.EnvironmentalDamage.Drowning.Times, TotalDamage = self.db.char.EnvironmentalDamage.Drowning.TotalDamage }
	result.Fatique = { Times = self.db.char.EnvironmentalDamage.Fatique.Times, TotalDamage = self.db.char.EnvironmentalDamage.Fatique.TotalDamage }
	result.Falling = { Times = self.db.char.EnvironmentalDamage.Falling.Times, TotalDamage = self.db.char.EnvironmentalDamage.Falling.TotalDamage }
	result.Fire = { Times = self.db.char.EnvironmentalDamage.Fire.Times, TotalDamage = self.db.char.EnvironmentalDamage.Fire.TotalDamage }
	result.Lava = { Times = self.db.char.EnvironmentalDamage.Lava.Times, TotalDamage = self.db.char.EnvironmentalDamage.Lava.TotalDamage }
	result.Slime = { Times = self.db.char.EnvironmentalDamage.Slime.Times, TotalDamage = self.db.char.EnvironmentalDamage.Slime.TotalDamage }
	result.TotalDeaths = self.db.char.Advancement.TotalDeaths;
	
	return result;
end;


-- *************************************
-- * GetFortuneStats
-- *************************************
function Historian:GetFortuneStats()
	local result = {};

	result.MostMoneyEver = self.db.char.Loot.MostMoneyEver;
	result.LootedCommon = self.db.char.Loot.ItemsOfQuality[1];
	result.LootedUncommon = self.db.char.Loot.ItemsOfQuality[2];
	result.LootedRare = self.db.char.Loot.ItemsOfQuality[3];
	result.LootedEpic = self.db.char.Loot.ItemsOfQuality[4];
	result.LootedLegendary = self.db.char.Loot.ItemsOfQuality[5];
	result.LootedArtifact = self.db.char.Loot.ItemsOfQuality[6];
	
	return result;
end;



-- *************************************
-- * GetAllKillStats
-- *************************************
function Historian:GetAllKillStats()
	return self.db.char.Kills.Mobs;
end;


-- *************************************
-- * GetAllSpellStats
-- *************************************
function Historian:GetAllSpellStats()
	return self.db.char.Spells.Casts;
end;


-- *************************************
-- * GetAllLevelStats
-- *************************************
function Historian:GetAllLevelStats()
	return self.db.char.Advancement.Levels;
end;


-- *************************************
-- * RecordPvPBattle
-- *************************************
function Historian:RecordPvPBattle(instance, stats)
	if (self.db.char.PvP[instance] == nil) then
		self.db.char.PvP[instance] = {};
	end;
	
	local now = date();
	self.db.char.PvP[instance][now] = {};
	
	tinsert(self.db.char.PvP[instance][now], stats);
	
	-- TODO: Clean up after X (100?) number of battles, so the WTF file doesn't get too big
end;